<?php
/**
 ------------------------------------------------------------------------
 SOLIDRES - Accommodation booking extension for Joomla
 ------------------------------------------------------------------------
 * @author    Solidres Team <contact@solidres.com>
 * @website   https://www.solidres.com
 * @copyright Copyright (C) 2013 - 2019 Solidres. All Rights Reserved.
 * @license   GNU General Public License version 3, or later
 ------------------------------------------------------------------------
 */

defined('_JEXEC') or die;

/**
 * Reservation Asset table
 *
 * @package       Solidres
 * @subpackage    ReservationAsset
 * @since         0.1.0
 */
class SolidresTableReservationAsset extends JTable
{
	public function __construct(JDatabaseDriver $db)
	{
		parent::__construct('#__sr_reservation_assets', 'id', $db);
		$this->setColumnAlias('published', 'state');

		if (class_exists('JTableObserverTags'))
		{
			JTableObserverTags::createObserver($this, ['typeAlias' => 'com_solidres.property']);
		}
		else
		{
			$this->typeAlias = 'com_solidres.property';
		}
	}

	/**
	 * Overloaded bind function to pre-process the params.
	 *
	 * @param    array $array Named array
	 * @param   string $ignore
	 *
	 * @return    null|string    null is operation was satisfactory, otherwise returns an error
	 * @see        JTable:bind
	 * @since      1.5
	 */
	public function bind($array, $ignore = '')
	{
		if (isset($array['params']) && is_array($array['params']))
		{
			$registry = new JRegistry();
			$registry->loadArray($array['params']);
			$array['params'] = (string) $registry;
		}

		if (isset($array['metadata']) && is_array($array['metadata']))
		{
			$registry = new JRegistry();
			$registry->loadArray($array['metadata']);
			$array['metadata'] = (string) $registry;
		}

		// Bind the rules.
		if (isset($array['rules']) && is_array($array['rules']))
		{
			$rules = new JAccessRules($array['rules']);
			$this->setRules($rules);
		}

		return parent::bind($array, $ignore);
	}

	/**
	 * Overloaded check function
	 *
	 * @return  boolean  True on success, false on failure
	 *
	 * @see         Table::check()
	 * @since       1.5
	 * @deprecated  3.1.4 Class will be removed upon completion of transition to UCM
	 */
	public function check()
	{
		$this->name  = htmlspecialchars_decode($this->name, ENT_QUOTES);
		$this->alias = JApplicationHelper::stringURLSafe($this->alias);

		if (empty ($this->alias))
		{
			$this->alias = JApplicationHelper::stringURLSafe($this->name);
		}

		if (empty ($this->geo_state_id))
		{
			$this->geo_state_id = null;
		}

		if (empty ($this->partner_id))
		{
			$this->partner_id = null;
			// Automatically assign partner_id if this asset is created in front end via hub dashboard
			if (SRPlugin::isEnabled('user') && SRPlugin::isEnabled('hub') && JFactory::getApplication()->isClient('site'))
			{
				$customerTable = JTable::getInstance('Customer', 'SolidresTable');
				$customerTable->load(array('user_id' => JFactory::getUser()->get('id')));
				$this->partner_id = $customerTable->id;
			}
		}

		if (empty ($this->category_id))
		{
			$this->category_id = null;
		}

		if (empty($this->id))
		{
			// Set ordering to the last item if not set
			if (empty($this->ordering))
			{
				$db    = JFactory::getDbo();
				$query = $db->getQuery(true);
				$query->clear();
				$query->select('MAX(ordering)')->from($db->quoteName('#__sr_reservation_assets'));
				$db->setQuery($query);
				$max = $db->loadResult();

				$this->ordering = $max + 1;
			}
		}

		// If tax_id is empty, then set it to null
		if ($this->tax_id == 0)
		{
			$this->tax_id = null;
		}

		return true;
	}

	/**
	 * Method to delete a row from the database table by primary key value.
	 *
	 * @param    mixed $pk An optional primary key value to delete.  If not set the
	 *                     instance property value is used.
	 *
	 * @return    boolean    True on success.
	 * @since    1.0
	 * @link     http://docs.joomla.org/JTable/delete
	 */
	public function delete($pk = null)
	{
		JModelLegacy::addIncludePath(JPATH_ADMINISTRATOR . '/components/com_solidres/models', 'SolidresModel');

		// Check to see if it contains any Room Types, if yes then notify user to delete all of its Room Type first
		$query = $this->_db->getQuery(true);
		$query->select('name')->from($this->_db->quoteName('#__sr_reservation_assets'))->where('id = ' . $pk);
		$this->_db->setQuery($query);
		$assetName = $this->_db->loadResult();

		$query->clear();
		$query->select('COUNT(id)')->from($this->_db->quoteName('#__sr_room_types'))->where('reservation_asset_id = ' . $pk);
		$this->_db->setQuery($query);
		$result = (int) $this->_db->loadResult();
		if ($result > 0)
		{
			$e = new JException(JText::sprintf('SR_ERROR_RESERVATION_CONTAIN_ROOM_TYPE', $assetName));
			$this->setError($e);

			return false;
		}

		// Take care of Reservation
		$query->clear();
		$query->update($this->_db->quoteName('#__sr_reservations'))
			->set($this->_db->quoteName('reservation_asset_id') . ' = NULL')
			->where($this->_db->quoteName('reservation_asset_id') . ' = ' . (int) $pk);
		$this->_db->setQuery($query)->execute();

		// Take care of media, if it has any, remove all of them
		$query->clear();
		$query->delete($this->_db->quoteName('#__sr_media_reservation_assets_xref'))->where('reservation_asset_id = ' . $pk);
		$this->_db->setQuery($query)->execute();

		// Take care of Extra
		$extrasModel = JModelLegacy::getInstance('Extras', 'SolidresModel', array('ignore_request' => true));
		$extraModel  = JModelLegacy::getInstance('Extra', 'SolidresModel', array('ignore_request' => true));
		$extrasModel->setState('filter.reservation_asset_id', $pk);
		$extras = $extrasModel->getItems();

		foreach ($extras as $extra)
		{
			$extraModel->delete($extra->id);
		}

		// Take care of Coupon
		$couponsModel = JModelLegacy::getInstance('Coupons', 'SolidresModel', array('ignore_request' => true));
		$couponModel  = JModelLegacy::getInstance('Coupon', 'SolidresModel', array('ignore_request' => true));
		$couponsModel->setState('filter.reservation_asset_id', $pk);
		$coupons = $couponsModel->getItems();

		foreach ($coupons as $coupon)
		{
			$couponModel->delete($coupon->id);
		}

		// Take care of Custom Fields
		$query->clear();
		$query->delete($this->_db->quoteName('#__sr_reservation_asset_fields'))->where('reservation_asset_id = ' . $pk);
		$this->_db->setQuery($query)->execute();

		if (SRPlugin::isEnabled('hub'))
		{
			// Take care of Themes
			$query->clear();
			$query->delete($this->_db->quoteName('#__sr_reservation_asset_theme_xref'))->where('reservation_asset_id = ' . $pk);
			$this->_db->setQuery($query)->execute();

			// Take care of Facilities
			$query->clear();
			$query->delete($this->_db->quoteName('#__sr_facility_reservation_asset_xref'))->where('reservation_asset_id = ' . $pk);
			$this->_db->setQuery($query)->execute();
		}

		// Take care of Limit Booking
		if (SRPlugin::isEnabled('limitbooking'))
		{
			\JTable::addIncludePath(\SRPlugin::getAdminPath('limitbooking') . '/tables');
			\JModelLegacy::addIncludePath(\SRPlugin::getAdminPath('limitbooking') . '/models', 'SolidresModel');
			$limitBookingsModel = JModelLegacy::getInstance('LimitBookings', 'SolidresModel', array('ignore_request' => true));
			$limitBookingModel  = JModelLegacy::getInstance('LimitBooking', 'SolidresModel', array('ignore_request' => true));
			$limitBookingsModel->setState('filter.reservation_asset_id', $pk);
			$limitBookings = $limitBookingsModel->getItems();

			foreach ($limitBookings as $limitBooking)
			{
				$limitBookingModel->delete($limitBooking->id);
			}
		}

		// Take care of Discount
		if (SRPlugin::isEnabled('discount'))
		{
			$discountsModel = JModelLegacy::getInstance('Discounts', 'SolidresModel', array('ignore_request' => true));
			$discountModel  = JModelLegacy::getInstance('Discount', 'SolidresModel', array('ignore_request' => true));
			$discountsModel->setState('filter.reservation_asset_id', $pk);
			$discounts = $discountsModel->getItems();

			foreach ($discounts as $discount)
			{
				$discountModel->delete($discount->id);
			}
		}

		// Take care of config data
		$dataKeys   = array(
			'payments/%',
			'sms/%',
			'plugins/googleanalytics/%',
			'plugins/facebook/%',
			'channelmanager/myallocator/myallocator_property_id'
		);
		$dataKeyStr = array();
		foreach ($dataKeys as $dataKey)
		{
			$dataKeyStr[] = 'data_key LIKE ' . $this->_db->quote($dataKey);
		}
		$query->clear();
		$query->delete($this->_db->quoteName('#__sr_config_data'))
			->where('scope_id = ' . $pk)
			->where('(' . implode(' OR ', $dataKeyStr) . ')');
		$this->_db->setQuery($query)->execute();

		if (SRPlugin::isEnabled('experience'))
		{
			$query->clear()
				->delete($this->_db->qn('#__sr_experience_asset_xref'))
				->where($this->_db->qn('reservation_asset_id') . ' = ' . (int) $pk);
			$this->_db->setQuery($query)
				->execute();
		}

		// Delete itself, finally
		return parent::delete($pk);
	}

	/**
	 * Method to store a row in the database from the JTable instance properties.
	 * If a primary key value is set the row with that primary key value will be
	 * updated with the instance property values.  If no primary key value is set
	 * a new row will be inserted into the database with the properties from the
	 * JTable instance.
	 *
	 * @param   boolean $updateNulls True to update fields even if they are null.
	 *
	 * @return  boolean  True on success.
	 *
	 * @link    http://docs.joomla.org/JTable/store
	 * @since   11.1
	 */
	public function store($updateNulls = true)
	{
		$date = JFactory::getDate();
		$user = JFactory::getUser();

		$this->modified_date = $date->toSql();
		$this->name          = str_replace('"', "'", $this->name);

		if ($this->id)
		{
			// Existing item
			$this->modified_by = $user->get('id');
		}
		else
		{
			if (!(int) $this->created_date)
			{
				$this->created_date = $date->toSql();
			}

			if (empty($this->created_by))
			{
				$this->created_by = $user->get('id');
			}
		}

		// Only 1 asset can be set as default
		if ($this->default == 1)
		{
			$query = $this->_db->getQuery(true)
				->update($this->_db->quoteName($this->_tbl))
				->set($this->_db->quoteName('default') . ' = 0');
			if ($this->id)
			{
				$query->where('id <> ' . $this->id);
			}
			$this->_db->setQuery($query)->execute();
		}

		unset($this->asset_id);

		$this->_trackAssets = false;

		return parent::store($updateNulls);
	}
}
