<?php
/**
 ------------------------------------------------------------------------
 SOLIDRES - Accommodation booking extension for Joomla
 ------------------------------------------------------------------------
 * @author    Solidres Team <contact@solidres.com>
 * @website   https://www.solidres.com
 * @copyright Copyright (C) 2013 - 2019 Solidres. All Rights Reserved.
 * @license   GNU General Public License version 3, or later
 ------------------------------------------------------------------------
 */

defined('_JEXEC') or die('Restricted access');

/**
 * Extras model
 *
 * @package       Solidres
 * @subpackage    Extra
 * @since         0.1.0
 */

use Joomla\Utilities\ArrayHelper;
use Joomla\CMS\Factory as CMSFactory;

class SolidresModelExtras extends JModelList
{
	/**
	 * Constructor.
	 *
	 * @param    array    An optional associative array of configuration settings.
	 *
	 * @see        JController
	 * @since      1.6
	 */
	public function __construct($config = array())
	{
		if (empty($config['filter_fields']))
		{
			$config['filter_fields'] = array(
				'id', 'a.id',
				'name', 'a.name',
				'state', 'a.state',
				'created_date', 'a.created_date',
				'created_by', 'a.created_by',
				'ordering', 'a.ordering',
				'price', 'a.price',
				'reservation_asset_id', 'a.reservation_asset_id',
			);
		}

		parent::__construct($config);
	}

	/**
	 * Method to auto-populate the model state.
	 *
	 * Note. Calling getState in this method will result in recursion.
	 *
	 * @since    1.6
	 */
	protected function populateState($ordering = 'a.name', $direction = 'asc')
	{
		// Load the filter state.
		$search = $this->getUserStateFromRequest($this->context . '.filter.search', 'filter_search');
		$this->setState('filter.search', $search);

		$published = $this->getUserStateFromRequest($this->context . '.filter.state', 'filter_state', '', 'string');
		$this->setState('filter.state', $published);

		$reservationAssetId = $this->getUserStateFromRequest($this->context . '.filter.reservation_asset_id', 'filter_reservation_asset_id', '');
		$this->setState('filter.reservation_asset_id', $reservationAssetId);

		$app = CMSFactory::getApplication();

		if ($app->input->getMethod() === 'POST')
		{
			$access = $app->input->post->get('access');
			$app->setUserState($this->context . '.filter.access', $access);
		}
		else
		{
			$access = $this->getUserStateFromRequest($this->context . '.filter.access', 'filter_access');
		}

		$this->setState('filter.access', $access);

		// List state information.
		parent::populateState($ordering, $direction);
	}

	public function getStoreId($id = '')
	{
		$id .= ':' . $this->getState('filter.search');
		$id .= ':' . $this->getState('filter.state');
		$id .= ':' . $this->getState('filter.reservation_asset_id');
		$id .= ':' . serialize($this->getState('filter.access'));

		return parent::getStoreId($id);
	}

	public function getItems()
	{
		$items = parent::getItems();

		if (!empty($items))
		{
			$showTaxIncl = $this->getState('filter.show_price_with_tax', 0);
			$taxTable    = JTable::getInstance('Tax', 'SolidresTable');

			// Caching to prevent unnecessary multiple loading
			$taxRates = array();

			foreach ($items as $item)
			{
				$solidresCurrency = new SRCurrency(0, $item->currency_id);

				if (isset($item->tax_id) && !isset($taxRates[$item->tax_id]))
				{
					$taxTable->load($item->tax_id);
					$taxRates[$item->tax_id] = $taxTable->rate;
				}

				$taxAmount      = 0;
				$taxAdultAmount = 0;
				$taxChildAmount = 0;
				if (isset($taxRates[$item->tax_id]) && $taxRates[$item->tax_id] > 0)
				{
					$taxAmount      = $item->price * $taxRates[$item->tax_id];
					$taxAdultAmount = $item->price_adult * $taxRates[$item->tax_id];
					$taxChildAmount = $item->price_child * $taxRates[$item->tax_id];
				}

				// For charge type != per person
				$item->currencyTaxIncl = clone $solidresCurrency;
				$item->currencyTaxExcl = clone $solidresCurrency;
				$item->currencyTaxIncl->setValue($item->price + $taxAmount);
				$item->currencyTaxExcl->setValue($item->price);
				$item->price_tax_incl = $item->price + $taxAmount;
				$item->price_tax_excl = $item->price;

				// For adult
				$item->currencyAdultTaxIncl = clone $solidresCurrency;
				$item->currencyAdultTaxExcl = clone $solidresCurrency;
				$item->currencyAdultTaxIncl->setValue($item->price_adult + $taxAdultAmount);
				$item->currencyAdultTaxExcl->setValue($item->price_adult);
				$item->price_adult_tax_incl = $item->price_adult + $taxAdultAmount;
				$item->price_adult_tax_excl = $item->price_adult;

				// For child
				$item->currencyChildTaxIncl = clone $solidresCurrency;
				$item->currencyChildTaxExcl = clone $solidresCurrency;
				$item->currencyChildTaxIncl->setValue($item->price_child + $taxChildAmount);
				$item->currencyChildTaxExcl->setValue($item->price_child);
				$item->price_child_tax_incl = $item->price_child + $taxChildAmount;
				$item->price_child_tax_excl = $item->price_child;

				if ($showTaxIncl)
				{
					$item->currency      = $item->currencyTaxIncl;
					$item->currencyAdult = $item->currencyAdultTaxIncl;
					$item->currencyChild = $item->currencyChildTaxIncl;
				}
				else
				{
					$item->currency      = $item->currencyTaxExcl;
					$item->currencyAdult = $item->currencyAdultTaxExcl;
					$item->currencyChild = $item->currencyChildTaxExcl;
				}
			}
		}

		return $items;
	}

	/**
	 * Build an SQL query to load the list data.
	 *
	 * @return    JDatabaseQuery
	 * @since    1.6
	 */
	protected function getListQuery()
	{
		$solidresConfig   = JComponentHelper::getParams('com_solidres');
		$numberOfDecimals = $solidresConfig->get('number_decimal_points', 2);
		$db               = $this->getDbo();
		$query            = $db->getQuery(true)
			->select($this->getState('list.select', 'a.id, a.name, a.state, a.description, a.created_date, a.modified_date, a.created_by, a.modified_by, ROUND(a.price, ' . $numberOfDecimals . ') AS price, ROUND(a.price_adult, ' . $numberOfDecimals . ') AS price_adult, ROUND(a.price_child, ' . $numberOfDecimals . ') AS price_child, a.ordering, a.max_quantity, a.daily_chargable, a.reservation_asset_id, a.experience_id, a.mandatory, a.charge_type, a.tax_id, a.params, a.scope, a.access'))
			->from($db->qn('#__sr_extras', 'a'));
		$query->select('a2.name AS reservationasset, a2.currency_id as currency_id')
			->leftJoin($db->qn('#__sr_reservation_assets', 'a2') . ' ON a.reservation_asset_id = a2.id')
			->where('a.scope = 0');

		// Join over the access groups.
		$query->select('ag.title AS access_level')
			->join('LEFT', '#__viewlevels AS ag ON ag.id = a.access');

		// Filter by published state
		$published = $this->getState('filter.state');
		if (is_numeric($published))
		{
			$query->where('a.state = ' . (int) $published);
		}
		else if ($published === '')
		{
			$query->where('(a.state IN (0, 1))');
		}

		// Filter by reservation asset.
		$reservationAssetId = $this->getState('filter.reservation_asset_id');
		if (is_numeric($reservationAssetId))
		{
			$query->where('a.reservation_asset_id = ' . (int) $reservationAssetId);
		}

		// If loading from front end, make sure we only load items belong to current user
		$isFrontEnd = JFactory::getApplication()->isClient('site');
		//$partnerId  = (int) $this->getState('filter.partner_id', 0);

		if ($isFrontEnd && ($partnerIds = SRUtilities::getPartnerIds()))
		{
			$query->where('a2.state = 1 AND a2.partner_id IN (' . join(',', $partnerIds) . ')');
		}

		// Filter by room type
		$roomTypeId = $this->getState('filter.room_type_id');
		if (is_numeric($roomTypeId))
		{
			$query->innerJoin($db->quoteName('#__sr_room_type_extra_xref') . '  as rxt ON a.id = rxt.extra_id AND rxt.room_type_id = ' . $db->quote($roomTypeId));
		}

		// Filter by charge type, support filter by multiple charge types
		$chargeType = $this->getState('filter.charge_type');
		$chargeType = (array) $chargeType;
		if (!empty($chargeType))
		{
			$query->where('a.charge_type IN (' . implode(',', $chargeType) . ')');
		}

		// Filter by mandatory
		$mandatory = $this->getState('filter.mandatory');
		if (is_numeric($mandatory))
		{
			$query->where('a.mandatory = ' . (int) $mandatory);
		}

		// Filter by search in title
		$search = $this->getState('filter.search');
		if (!empty($search))
		{
			if (stripos($search, 'id:') === 0)
			{
				$query->where('a.id = ' . (int) substr($search, 3));
			}
			else
			{
				$search = $db->Quote('%' . $db->escape($search, true) . '%');
				$query->where('a.name LIKE ' . $search);
			}
		}

		// Filter by access level.
		$access = $this->getState('filter.access');

		if (is_numeric($access))
		{
			$query->where('a.access = ' . (int) $access);
		}
		elseif (is_array($access))
		{
			$access = ArrayHelper::toInteger($access);
			$access = join(',', $access);
			$query->where('a.access IN (' . $access . ')');
		}

		// Filter by user access level.
		$user = CMSFactory::getUser();

		if (!$user->authorise('core.admin'))
		{
			$groups = implode(',', $user->getAuthorisedViewLevels());
			$query->where('a.access IN (' . $groups . ')');
		}

		// Add the list ordering clause.
		$orderCol  = $this->state->get('list.ordering', 'a.ordering');
		$orderDirn = $this->state->get('list.direction', 'ASC');

		$query->order($db->escape($orderCol) . ' ' . $db->escape($orderDirn));

		return $query;
	}
}
