<?php
/**
 ------------------------------------------------------------------------
 SOLIDRES - Accommodation booking extension for Joomla
 ------------------------------------------------------------------------
 * @author    Solidres Team <contact@solidres.com>
 * @website   https://www.solidres.com
 * @copyright Copyright (C) 2013 - 2019 Solidres. All Rights Reserved.
 * @license   GNU General Public License version 3, or later
 ------------------------------------------------------------------------
 */

defined('_JEXEC') or die;

/**
 * Reservation handler class
 *
 * @package       Solidres
 * @subpackage    Reservation
 */
class SRReservation
{
	/**
	 * The database object
	 *
	 * @var object
	 */
	protected $_dbo = null;

	public function __construct()
	{
		$this->_dbo = JFactory::getDbo();
	}

	/**
	 * Generate unique string for Reservation
	 *
	 * @param string $srcString The string that need to be calculate checksum
	 *
	 * @return string The unique string for each Reservation
	 */
	public function getCode($srcString)
	{
		return hash('crc32', (string) $srcString . uniqid());
	}

	/**
	 * Check a room to see if it is allowed to be booked in the period from $checkin -> $checkout
	 *
	 * @param int    $roomId
	 * @param string $checkin
	 * @param string $checkout
	 * @param int    $bookingType 0 is booking per night and 1 is booking per day
	 * @param int    $excludeId
	 * @param int    $confirmationState
	 *
	 * @return boolean  True if the room is ready to be booked, False otherwise
	 */
	public function isRoomAvailable($roomId = 0, $checkin, $checkout, $bookingType = 0, $excludeId = 0, $confirmationState = 5)
	{
		$checkinQuote  = $this->_dbo->quote($checkin);
		$checkoutQuote = $this->_dbo->quote($checkout);

		$query = $this->_dbo->getQuery(true);
		$query->select('checkin, checkout');
		$query->from($this->_dbo->quoteName('#__sr_reservations') . ' as res');

		$query->join('INNER', $this->_dbo->quoteName('#__sr_reservation_room_xref') . ' as room
									ON res.id = room.reservation_id
									AND room.room_id = ' . $this->_dbo->quote($roomId) . ($excludeId > 0 ? ' AND res.id != ' . (int) $excludeId : ''));
		//$query->where('res.checkout >= '. $this->_dbo->quote(date('Y-m-d')));
		$query->where("(
									(res.checkin BETWEEN $checkinQuote AND $checkoutQuote OR res.checkout BETWEEN $checkinQuote AND $checkoutQuote)
									OR
									($checkinQuote BETWEEN res.checkin AND res.checkout OR $checkoutQuote BETWEEN res.checkin AND res.checkout)
								)");
		$query->where('res.state = ' . (int) $confirmationState);
		$query->order('res.checkin');

		$this->_dbo->setQuery($query);
		$result = $this->_dbo->loadObjectList();

		if (is_array($result))
		{
			foreach ($result as $currentReservation)
			{
				$currentCheckin  = $currentReservation->checkin;
				$currentCheckout = $currentReservation->checkout;

				if ($checkin != $checkout) // Normal check
				{
					if (0 == $bookingType) // Per night
					{
						if
						(
							($checkin <= $currentCheckin && $checkout > $currentCheckin) ||
							($checkin >= $currentCheckin && $checkout <= $currentCheckout) ||
							($checkin < $currentCheckout && $checkout >= $currentCheckout)
						)
						{
							return false;
						}
					}
					else // Per day
					{
						if
						(
							($checkin <= $currentCheckin && $checkout >= $currentCheckin) ||
							($checkin >= $currentCheckin && $checkout <= $currentCheckout) ||
							($checkin <= $currentCheckout && $checkout >= $currentCheckout)
						)
						{
							return false;
						}
					}
				}
				else // Edge case when we check for available of a single date (checkin == checkout)
				{
					if (0 == $bookingType) // Per night
					{
						if
						(
							($checkin <= $currentCheckin && $checkout > $currentCheckin) ||
							($checkin >= $currentCheckin && $checkout < $currentCheckout) ||
							($checkin < $currentCheckout && $checkout >= $currentCheckout)
						)
						{
							return false;
						}
					}
					else
					{
						if
						(
							($checkin <= $currentCheckin && $checkout > $currentCheckin) ||
							($checkin >= $currentCheckin && $checkout < $currentCheckout) ||
							($checkin <= $currentCheckout && $checkout >= $currentCheckout)
						)
						{
							return false;
						}
					}
				}
			}
		}

		return true;
	}

	/**
	 * Check a room to see if it is allowed to be booked in the period from $checkin -> $checkout
	 *
	 * @param int    $roomId
	 * @param string $checkin
	 * @param string $checkout
	 * @param int    $bookingType 0 is booking per night and 1 is booking per day
	 * @param int    $excludeId
	 *
	 * @return boolean  True if the room is blocked, False otherwise
	 *
	 * @since 2.1.0
	 */
	public function isRoomLimited($roomId = 0, $checkin, $checkout, $bookingType = 0, $excludeId = 0)
	{
		if (!SRPlugin::isEnabled('limitbooking'))
		{
			return false;
		}

		$query = $this->_dbo->getQuery(true);

		$checkinMySQLFormat  = "STR_TO_DATE(" . $this->_dbo->quote($checkin) . ", '%Y-%m-%d')";
		$checkoutMySQLFormat = "STR_TO_DATE(" . $this->_dbo->quote($checkout) . ", '%Y-%m-%d')";

		if (0 == $bookingType) // Booking per night
		{
			$query->select('COUNT(*) FROM ' . $this->_dbo->quoteName('#__sr_limit_booking_details') . '
									WHERE room_id = ' . $this->_dbo->quote($roomId) . ' AND 
									limit_booking_id IN (SELECT id FROM ' . $this->_dbo->quoteName('#__sr_limit_bookings') . '
									WHERE
									(
										(' . $checkinMySQLFormat . ' <= start_date AND ' . $checkoutMySQLFormat . ' > start_date )
										OR
										(' . $checkinMySQLFormat . ' >= start_date AND ' . $checkoutMySQLFormat . ' <= end_date )
										OR
										(start_date != end_date AND ' . $checkinMySQLFormat . ' <= end_date AND ' . $checkoutMySQLFormat . ' >= end_date )
										OR
										(start_date = end_date AND ' . $checkinMySQLFormat . ' <= end_date AND ' . $checkoutMySQLFormat . ' > end_date )
									)
									AND state = 1 ' . ($excludeId > 0 ? ' AND id != ' . (int) $excludeId : '') . '
									)');
		}
		else // Booking per day
		{
			$query->select('COUNT(*) FROM ' . $this->_dbo->quoteName('#__sr_limit_booking_details') . '
									WHERE room_id = ' . $this->_dbo->quote($roomId) . ' AND 
									limit_booking_id IN (SELECT id FROM ' . $this->_dbo->quoteName('#__sr_limit_bookings') . '
									WHERE
									(
										(' . $checkinMySQLFormat . ' <= start_date AND ' . $checkoutMySQLFormat . ' >= start_date )
										OR
										(' . $checkinMySQLFormat . ' >= start_date AND ' . $checkoutMySQLFormat . ' <= end_date )
										OR
										(' . $checkinMySQLFormat . ' <= end_date AND ' . $checkoutMySQLFormat . ' >= end_date )
									)
									AND state = 1 ' . ($excludeId > 0 ? ' AND id != ' . (int) $excludeId : '') . '
									)');
		}

		$this->_dbo->setQuery($query);

		if ($this->_dbo->loadResult() > 0)
		{
			return true;
		}

		return false;
	}

	/**
	 * Store reservation data and related data like children ages or other room preferences
	 *
	 * @param   int   $reservationId
	 * @param   array $room Room information
	 *
	 * @return void
	 */
	public function storeRoom($reservationId, $room)
	{
		$query = $this->_dbo->getQuery(true);

		// Store main room info
		$query->insert($this->_dbo->quoteName('#__sr_reservation_room_xref'));
		$columns = array(
			$this->_dbo->quoteName('reservation_id'),
			$this->_dbo->quoteName('room_id'),
			$this->_dbo->quoteName('room_label'),
			$this->_dbo->quoteName('adults_number'),
			$this->_dbo->quoteName('children_number'),
			$this->_dbo->quoteName('guest_fullname'),
			$this->_dbo->quoteName('room_price'),
			$this->_dbo->quoteName('room_price_tax_incl'),
			$this->_dbo->quoteName('room_price_tax_excl')
		);

		if (isset($room['tariff_id']) && !is_null($room['tariff_id']))
		{
			$columns = array_merge($columns, array(
				$this->_dbo->quoteName('tariff_id'),
				$this->_dbo->quoteName('tariff_title'),
				$this->_dbo->quoteName('tariff_description')
			));
		}

		$query->columns($columns);

		$values = (int) $reservationId . ',' .
			$this->_dbo->quote($room['room_id']) . ',' .
			$this->_dbo->quote($room['room_label']) . ',' .
			$this->_dbo->quote(isset($room['adults_number']) ? $room['adults_number'] : 0) . ',' .
			$this->_dbo->quote(isset($room['children_number']) ? $room['children_number'] : 0) . ',' .
			$this->_dbo->quote(isset($room['guest_fullname']) ? $room['guest_fullname'] : '') . ',' .
			$this->_dbo->quote(isset($room['room_price']) ? $room['room_price'] : 0) . ',' .
			$this->_dbo->quote(isset($room['room_price_tax_incl']) ? $room['room_price_tax_incl'] : 0) . ',' .
			$this->_dbo->quote(isset($room['room_price_tax_excl']) ? $room['room_price_tax_excl'] : 0);

		if (isset($room['tariff_id']) && !is_null($room['tariff_id']))
		{
			$values .= ',' . $this->_dbo->quote($room['tariff_id']) . ',' .
				$this->_dbo->quote($room['tariff_title']) . ',' .
				$this->_dbo->quote($room['tariff_description']);
		}

		$query->values($values);

		$this->_dbo->setQuery($query);
		$this->_dbo->execute();

		// Store related data
		$recentInsertedId = $this->_dbo->insertid();

		if (isset($room['children_number']) && isset($room['children_ages']))
		{
			for ($i = 0; $i < $room['children_number']; $i++)
			{
				$query->clear();
				$query->insert($this->_dbo->quoteName('#__sr_reservation_room_details'));
				$query->columns(array(
					$this->_dbo->quoteName('reservation_room_id'),
					$this->_dbo->quoteName('key'),
					$this->_dbo->quoteName('value')
				));
				$query->values(
					(int) $recentInsertedId . ',' .
					$this->_dbo->quote('child' . ($i + 1)) . ',' .
					$this->_dbo->quote($room['children_ages'][$i])
				);

				$this->_dbo->setQuery($query);
				$this->_dbo->execute();
			}
		}

		if (isset($room['preferences']))
		{
			foreach ($room['preferences'] as $key => $value)
			{
				$query->clear();
				$query->insert($this->_dbo->quoteName('#__sr_reservation_room_details'));
				$query->columns(array(
					$this->_dbo->quoteName('reservation_room_id'),
					$this->_dbo->quoteName('key'),
					$this->_dbo->quoteName('value')
				));
				$query->values(
					(int) $recentInsertedId . ',' .
					$this->_dbo->quote($key) . ',' .
					$this->_dbo->quote($value)
				);

				$this->_dbo->setQuery($query);
				$this->_dbo->execute();
			}
		}
	}

	/**
	 * Store extra information
	 *
	 * @param  int    $reservationId
	 * @param  int    $roomId
	 * @param  string $roomLabel
	 * @param  int    $extraId
	 * @param  string $extraName
	 * @param  int    $extraQuantity The extra quantity or NULL if extra does not have quantity
	 * @param  int    $price
	 *
	 * @return void
	 */
	public function storeExtra($reservationId, $roomId, $roomLabel, $extraId, $extraName, $extraQuantity = null, $price = 0)
	{
		$query = $this->_dbo->getQuery(true);
		$query->insert($this->_dbo->quoteName('#__sr_reservation_room_extra_xref'));
		$query->columns(array(
			$this->_dbo->quoteName('reservation_id'),
			$this->_dbo->quoteName('room_id'),
			$this->_dbo->quoteName('room_label'),
			$this->_dbo->quoteName('extra_id'),
			$this->_dbo->quoteName('extra_name'),
			$this->_dbo->quoteName('extra_quantity'),
			$this->_dbo->quoteName('extra_price')
		));
		$query->values(
			$this->_dbo->quote($reservationId) . ',' .
			$this->_dbo->quote($roomId) . ',' .
			$this->_dbo->quote($roomLabel) . ',' .
			$this->_dbo->quote($extraId) . ',' .
			$this->_dbo->quote($extraName) . ',' .
			($extraQuantity === null ? null : $this->_dbo->quote($extraQuantity)) . ',' .
			$this->_dbo->quote($price)
		);
		$this->_dbo->setQuery($query);
		$this->_dbo->execute();
	}

	/**
	 * Check for the validity of check in and check out date
	 *
	 * Conditions
	 * + Number of days a booking must be made in advance
	 * + Maximum length of stay
	 *
	 * @param string $checkIn
	 * @param string $checkOut
	 * @param array  $conditions
	 *
	 * @throws Exception
	 * @return Boolean
	 */
	public function isCheckInCheckOutValid($checkIn, $checkOut, $conditions)
	{
		$checkIn  = new DateTime($checkIn);
		$checkOut = new DateTime($checkOut);
		$today    = new DateTime(date('Y-m-d'));

		if ($checkIn < $today || $checkOut < $today)
		{
			throw new Exception('SR_ERROR_PAST_CHECKIN_CHECKOUT', 50005);
		}

		if ((!isset($conditions['booking_type']) && (isset($conditions['min_length_of_stay']) && $conditions['min_length_of_stay'] > 0))
			||
			(isset($conditions['booking_type']) && $conditions['booking_type'] == 0)
		)
		{
			if ($checkOut <= $checkIn)
			{
				throw new Exception('SR_ERROR_INVALID_CHECKIN_CHECKOUT', 50001);
			}
		}

		// Interval between check in and check out date
		if (isset($conditions['min_length_of_stay']) && $conditions['min_length_of_stay'] > 0)
		{
			$interval1 = $checkOut->diff($checkIn)->format('%a');

			if ($interval1 < $conditions['min_length_of_stay']) // count nights, not days
			{
				throw new Exception('SR_ERROR_INVALID_MIN_LENGTH_OF_STAY_' . $conditions['booking_type'], 50002);
			}
		}

		// Interval between checkin and today
		$interval2 = $checkIn->diff($today)->format('%a');

		if (isset($conditions['min_days_book_in_advance']) && $conditions['min_days_book_in_advance'] > 0)
		{
			if ($interval2 < $conditions['min_days_book_in_advance'])
			{
				throw new Exception('SR_ERROR_INVALID_MIN_DAYS_BOOK_IN_ADVANCE', 50003);
			}
		}

		if (isset($conditions['max_days_book_in_advance']) && $conditions['max_days_book_in_advance'] > 0)
		{
			if ($interval2 > $conditions['max_days_book_in_advance'])
			{
				throw new Exception('SR_ERROR_INVALID_MAX_DAYS_BOOK_IN_ADVANCE', 50004);
			}
		}

		return true;
	}

	/**
	 * Send email when reservation is completed
	 *
	 * @since  0.1.0
	 *
	 * @param int    $reservationId The reservation to get the reservation info for emails (Optional)
	 * @param string $state         The target reservation state that trigger sending email, at this moment only
	 *                              cancellation state is taken into consideration
	 *
	 * @return boolean True if email sending completed successfully. False otherwise
	 */
	public function sendEmail($reservationId = null, $state = '')
	{
		$lang = JFactory::getLanguage();
		$lang->load('com_solidres', JPATH_ADMINISTRATOR . '/components/com_solidres', null, true);
		$lang->load('com_solidres', JPATH_SITE . '/components/com_solidres', null, true);

		$subject           = array();
		$body              = array();
		$solidresConfig    = JComponentHelper::getParams('com_solidres');
		$config            = JFactory::getConfig();
		$app               = JFactory::getApplication();
		$dateFormat        = $solidresConfig->get('date_format', 'd-m-Y');
		$cancellationState = $solidresConfig->get('cancel_state', 4);
		$tzoffset          = $config->get('offset');
		$timezone          = new DateTimeZone($tzoffset);
		$context           = 'com_solidres.reservation.process';

		$layout = SRLayoutHelper::getInstance();

		if (isset($reservationId))
		{
			$savedReservationId = $reservationId;
		}
		else
		{
			$savedReservationId = $app->getUserState($context . '.savedReservationId');
		}

		JModelLegacy::addIncludePath(JPATH_ADMINISTRATOR . '/components/com_solidres/models', 'SolidresModel');
		JTable::addIncludePath(JPATH_ADMINISTRATOR . '/components/com_solidres/tables', 'SolidresTable');
		$modelReservation     = JModelLegacy::getInstance('Reservation', 'SolidresModel', array('ignore_request' => true));
		$modelAsset           = JModelLegacy::getInstance('ReservationAsset', 'SolidresModel', array('ignore_request' => true));
		$savedReservationData = $modelReservation->getItem($savedReservationId);
		$asset                = $modelAsset->getItem($savedReservationData->reservation_asset_id);
		$stayLength           = (int) SRUtilities::calculateDateDiff($savedReservationData->checkin, $savedReservationData->checkout);
		$direction            = JFactory::getDocument()->direction;

		// Load override language file
		$lang->load('com_solidres_category_' . $asset->category_id, JPATH_COMPONENT, null, true);

		$customerEmail    = $savedReservationData->customer_email;
		$hotelEmailList[] = $asset->email;
		$enableTouristTax = false;
		if (isset($asset->params['enable_tourist_tax']))
		{
			$enableTouristTax = $asset->params['enable_tourist_tax'];
		}

		switch ($state)
		{
			case $cancellationState:
				//$subject[$customerEmail] = JText::_('SR_EMAIL_RESERVATION_CANCELLED');
				$subject[$asset->email] = JText::sprintf(
					'SR_EMAIL_RESERVATION_CANCELLED_NOTIFICATION',
					$savedReservationData->code,
					$savedReservationData->customer_firstname,
					$savedReservationData->customer_lastname
				);

				break;
			case ''; // When new reservation is created for the first time
			default:
				//$subject[$customerEmail] = JText::_('SR_EMAIL_RESERVATION_COMPLETE');
				$subject[$asset->email] = JText::sprintf(
					'SR_EMAIL_NEW_RESERVATION_NOTIFICATION',
					$savedReservationData->code,
					$savedReservationData->customer_firstname,
					$savedReservationData->customer_lastname
				);
				break;
		}

		// If User plugin is installed and enabled
		if (SRPlugin::isEnabled('user') && SRPlugin::isEnabled('hub') && !is_null($asset->partner_id))
		{
			JModelLegacy::addIncludePath(SRPlugin::getAdminPath('user') . '/models', 'SolidresModel');
			$modelCustomer = JModelLegacy::getInstance('Customer', 'SolidresModel', array('ignore_request' => true));
			$partner       = $modelCustomer->getItem($asset->partner_id);
			if (!empty($partner->email) && $partner->email != $asset->email)
			{
				$hotelEmailList[]         = $partner->email;
				$subject[$partner->email] = $subject[$asset->email];
			}
		}

		$bankWireInstructions = array();
		if ($savedReservationData->payment_method_id == 'bankwire')
		{
			$solidresPaymentConfigData               = new SRConfig(array('scope_id' => $savedReservationData->reservation_asset_id));
			$bankWireInstructions['account_name']    = SRUtilities::translateText($solidresPaymentConfigData->get('payments/bankwire/bankwire_accountname'));
			$bankWireInstructions['account_details'] = SRUtilities::translateText($solidresPaymentConfigData->get('payments/bankwire/bankwire_accountdetails'));
		}

		// Prepare some costs data to be showed
		$baseCurrency = new SRCurrency(0, $savedReservationData->currency_id);
		$subTotal     = clone $baseCurrency;
		$subTotal->setValue($savedReservationData->total_price_tax_excl);

		$discountTotal = clone $baseCurrency;
		$discountTotal->setValue($savedReservationData->total_discount);

		$tax = clone $baseCurrency;
		$tax->setValue($savedReservationData->tax_amount);
		$touristTax = clone $baseCurrency;
		$touristTax->setValue($savedReservationData->tourist_tax_amount);

		$paymentMethodSurcharge = clone $baseCurrency;
		$paymentMethodSurcharge->setValue($savedReservationData->payment_method_surcharge);
		$paymentMethodDiscount = clone $baseCurrency;
		$paymentMethodDiscount->setValue($savedReservationData->payment_method_discount);

		$totalExtraPriceTaxExcl = clone $baseCurrency;
		$totalExtraPriceTaxExcl->setValue($savedReservationData->total_extra_price_tax_excl);
		$extraTax = clone $baseCurrency;
		$extraTax->setValue($savedReservationData->total_extra_price_tax_incl - $savedReservationData->total_extra_price_tax_excl);

		$grandTotal = clone $baseCurrency;
		if ($savedReservationData->discount_pre_tax)
		{
			$grandTotalAmount = $savedReservationData->total_price_tax_excl - $savedReservationData->total_discount + $savedReservationData->tax_amount + $savedReservationData->total_extra_price;
		}
		else
		{
			$grandTotalAmount = $savedReservationData->total_price_tax_excl + $savedReservationData->tax_amount - $savedReservationData->total_discount + $savedReservationData->total_extra_price;
		}
		$grandTotalAmount += isset($savedReservationData->tourist_tax_amount) ? $savedReservationData->tourist_tax_amount : 0;
		$grandTotalAmount += isset($savedReservationData->payment_method_surcharge) ? $savedReservationData->payment_method_surcharge : 0;
		$grandTotalAmount -= isset($savedReservationData->payment_method_discount) ? $savedReservationData->payment_method_discount : 0;
		$grandTotal->setValue($grandTotalAmount);

		$depositAmount = clone $baseCurrency;
		$depositAmount->setValue(isset($savedReservationData->deposit_amount) ? $savedReservationData->deposit_amount : 0);

		$totalPaid = clone $baseCurrency;
		$totalPaid->setValue(isset($savedReservationData->total_paid) ? $savedReservationData->total_paid : 0);

		$dueAmount = clone $baseCurrency;
		$dueAmount->setValue($grandTotalAmount - (isset($savedReservationData->total_paid) ? $savedReservationData->total_paid : 0));

		// Get custom field if available
		$reservationCustomerField = null;
		if (SRPlugin::isEnabled('customfield'))
		{
			$customFieldContext      = 'com_solidres.customer.' . $savedReservationData->id;
			$reservationCustomFields = SRCustomFieldHelper::getValues(array('context' => $customFieldContext), $savedReservationData->customer_language);
			require_once SRPlugin::getAdminPath('customfield') . '/helpers/customfieldvalue.php';
			$reservationCustomerField = new SRCustomFieldValue($reservationCustomFields);
		}

		$displayData = array(
			'reservation'                     => $savedReservationData,
			'reservationCustomerField'        => $reservationCustomerField,
			'subTotal'                        => $subTotal->format(),
			'totalDiscount'                   => $savedReservationData->total_discount > 0.00 ? $discountTotal->format() : null,
			'tax'                             => $tax->format(),
			'touristTax'                      => $touristTax->format(),
			'totalExtraPriceTaxExcl'          => $totalExtraPriceTaxExcl->format(),
			'extraTax'                        => $extraTax->format(),
			'grandTotal'                      => $grandTotal->format(),
			'stayLength'                      => $stayLength,
			'depositAmount'                   => $depositAmount->format(),
			'totalPaid'                       => $totalPaid->format(),
			'dueAmount'                       => $dueAmount->format(),
			'bankwireInstructions'            => $bankWireInstructions,
			'asset'                           => $asset,
			'dateFormat'                      => $dateFormat,
			'timezone'                        => $timezone,
			'baseCurrency'                    => $baseCurrency,
			'paymentMethodLabel'              => JText::_('SR_PAYMENT_METHOD_' . $savedReservationData->payment_method_id),
			'paymentMethodCustomEmailContent' => $app->getUserState($context . '.payment_method_custom_email_content', ''),
			'discountPreTax'                  => $savedReservationData->discount_pre_tax,
			'direction'                       => $direction,
			'enableTouristTax'                => $enableTouristTax,
			'paymentMethodSurcharge'          => $paymentMethodSurcharge->format(),
			'paymentMethodDiscount'           => $paymentMethodDiscount->format(),
			'layout'                          => $layout
		);

		// Send email to customer
		$customerEmailLang = !empty($savedReservationData->customer_language) ? $savedReservationData->customer_language : null;
		$lang->load('com_solidres', JPATH_ADMINISTRATOR . '/components/com_solidres', $customerEmailLang, true);
		$lang->load('com_solidres', JPATH_SITE . '/components/com_solidres', $customerEmailLang, true);
		$lang->load('com_solidres_category_' . $asset->category_id, JPATH_COMPONENT, $customerEmailLang, true);

		switch ($state)
		{
			case $cancellationState:
				$subject[$customerEmail]     = JText::_('SR_EMAIL_RESERVATION_CANCELLED');
				$displayData['greetingText'] = JText::sprintf('SR_EMAIL_GREETING_TEXT_CANCELLED', $savedReservationData->code, $displayData['asset']->name);
				break;
			default:
				$subject[$customerEmail]     = JText::_('SR_EMAIL_RESERVATION_COMPLETE');
				$displayData['greetingText'] = JText::sprintf('SR_EMAIL_GREETING_TEXT', $displayData['asset']->name);
				break;
		}

		$displayData['forceLang'] = $customerEmailLang;

		$body[$customerEmail] = $layout->render(
			'emails.reservation_complete_customer_html_inliner',
			$displayData
		);

		$mail = JFactory::getMailer();
		$mail->setSender(array($config->get('mailfrom'), $config->get('fromname')));
		$mail->addRecipient($customerEmail);
		$mail->setSubject($subject[$customerEmail]);
		$mail->setBody($body[$customerEmail]);
		$mail->IsHTML(true);

		if (SRPlugin::isEnabled('invoice'))
		{
			$invoiceTable = JTable::getInstance('Invoice', 'SolidresTable');
			$invoiceTable->load(array('reservation_id' => $savedReservationId));
			$layout->addIncludePath(array(
				SRPlugin::getPluginPath('invoice') . '/layouts',
				JPATH_BASE . '/templates/' . $app->getTemplate() . '/html/layouts/com_solidres'
			));
			$pdf = null;
			$pdf = $layout->render(
				'emails.reservation_complete_customer_pdf',
				$displayData,
				false
			);

			if ($solidresConfig->get('enable_pdf_attachment', 1) == 1)
			{
				$this->getPDFAttachment($mail, $pdf, $savedReservationId, $savedReservationData->code);
			}

			$selectedPaymentMethod    = $app->getUserState($context . '.payment_method_id', '');
			$autoSendPaymentMethods   = $solidresConfig->get('auto_send_invoice_payment_methods', '');
			$sendInvoiceAutomatically = ($solidresConfig->get('auto_create_invoice', 0) == 1 &&
				$solidresConfig->get('auto_send_invoice', 0) == 1
				&& in_array($selectedPaymentMethod, $autoSendPaymentMethods));
			if ($sendInvoiceAutomatically)
			{
				JPluginHelper::importPlugin('solidres');
				$invoiceFolder      = JPATH_ROOT . '/media/com_solidres/invoices/';
				$attachmentFileName = $solidresConfig->get('solidres_invoice_pdf_file_name', 'Invoice');
				if ($invoiceTable->id)
				{
					$mail->addAttachment(
						$invoiceFolder . $invoiceTable->filename,
						$attachmentFileName . '_' . $invoiceTable->invoice_number . '.pdf', 'base64', 'application/pdf'
					);
				}
			}
		}

		if (!$mail->send())
		{
			return false;
		}
		else
		{
			if (SRPlugin::isEnabled('invoice') && $sendInvoiceAutomatically)
			{
				$invoiceTable->set('sent_on', JFactory::getDate()->toSql());
				$invoiceTable->store();
			}
		}

		// Send to the hotel owner
		$lang->load('com_solidres', JPATH_ADMINISTRATOR . '/components/com_solidres', null, true);
		$lang->load('com_solidres', JPATH_SITE . '/components/com_solidres', null, true);
		$lang->load('com_solidres_category_' . $asset->category_id, JPATH_COMPONENT, null, true);
		$editLinks = array(
			'admin'   => JUri::root() . 'administrator/index.php?option=com_solidres&view=reservation&layout=edit&id=' . $displayData['reservation']->id,
			'partner' => JUri::root() . 'index.php?option=com_solidres&task=reservationform.edit&id=' . $displayData['reservation']->id
		);

		$displayData['editLink']  = $editLinks['admin'];
		$displayData['forceLang'] = null;
		switch ($state)
		{
			case $cancellationState:
				$displayData['greetingText'] = JText::sprintf('SR_EMAIL_NOTIFICATION_GREETING_TEXT_CANCELLED', $displayData['reservation']->code, $editLinks['admin']);
				break;
			default:
				$displayData['greetingText'] = JText::sprintf('SR_EMAIL_NOTIFICATION_GREETING_TEXT', $editLinks['admin']);
				break;
		}

		$body[$asset->email] = $layout->render(
			'emails.reservation_complete_owner_html_inliner',
			$displayData
		);

		$mail2 = JFactory::getMailer();
		$mail2->setSender(array($config->get('mailfrom'), $config->get('fromname')));
		$mail2->addRecipient($asset->email);
		$additionalNotificationEmails = isset($asset->params['additional_notification_emails']) ? explode(',', $asset->params['additional_notification_emails']) : array();
		if (!empty($additionalNotificationEmails))
		{
			$mail2->addRecipient($additionalNotificationEmails);
		}
		$mail2->setSubject($subject[$asset->email]);
		$mail2->setBody($body[$asset->email]);
		$mail2->IsHTML(true);

		if (!$mail2->send())
		{
			return false;
		}

		// Send to the hotel partner
		if (SRPlugin::isEnabled('user') && SRPlugin::isEnabled('hub') && isset($partner->email))
		{
			switch ($state)
			{
				case $cancellationState:
					$displayData['greetingText'] = JText::sprintf('SR_EMAIL_NOTIFICATION_GREETING_TEXT_CANCELLED', $displayData['reservation']->code, $editLinks['partner']);
					break;
				default:
					$displayData['greetingText'] = JText::sprintf('SR_EMAIL_NOTIFICATION_GREETING_TEXT', $editLinks['partner']);
					break;
			}

			$displayData['editLink'] = $editLinks['partner'];
			$body[$partner->email]   = $layout->render(
				'emails.reservation_complete_owner_html_inliner',
				$displayData
			);

			$mail3 = JFactory::getMailer();
			$mail3->setSender(array($config->get('mailfrom'), $config->get('fromname')));
			$mail3->addRecipient($partner->email);
			$mail3->setSubject($subject[$partner->email]);
			$mail3->setBody($body[$partner->email]);
			$mail3->IsHTML(true);

			if (!$mail3->send())
			{
				return false;
			}
		}

		return true;
	}

	/**
	 * Create PDF attachment.
	 *
	 * @param $mail        mail object.
	 * @param $reid        reservation id.
	 * @param $reCode      reservation code.
	 *
	 * @since 1.0.0
	 */
	protected function getPDFAttachment($mail, $content, $reid, $reCode)
	{
		JPluginHelper::importPlugin('solidres');
		$solidresConfig     = JComponentHelper::getParams('com_solidres');
		$attachmentFileName = $solidresConfig->get('solidres_voucher_pdf_file_name', 'voucher');
		$results            = JFactory::getApplication()->triggerEvent('onSolidresReservationEmail', array($content, $reid));

		if ($results)
		{
			$mail->addAttachment($results[0], $attachmentFileName . '_' . $reCode . '.pdf', 'base64', 'application/pdf');
		}
	}

	/**
	 * Send email when reservation is completed
	 *
	 * @since  2.5.0
	 *
	 * @return boolean True if email sending completed successfully. False otherwise
	 */
	public function sendBookingInquiryNotificationEmail($reservationId)
	{
		JModelLegacy::addIncludePath(JPATH_ADMINISTRATOR . '/components/com_solidres/models', 'SolidresModel');
		JLoader::register('SRUtilities', SRPATH_LIBRARY . '/utilities/utilities.php');
		$modelReservation = JModelLegacy::getInstance('Reservation', 'SolidresModel', array('ignore_request' => true));
		$modelAsset       = JModelLegacy::getInstance('ReservationAsset', 'SolidresModel', array('ignore_request' => true));
		$reservation      = $modelReservation->getItem($reservationId);
		$asset            = $modelAsset->getItem($reservation->reservation_asset_id);
		$mail             = JFactory::getMailer();
		$config           = JFactory::getConfig();

		$recipients                   = array();
		$recipients[]                 = $asset->email;
		$additionalNotificationEmails = isset($asset->params['additional_notification_emails']) ? explode(',', $asset->params['additional_notification_emails']) : array();
		if (!empty($additionalNotificationEmails))
		{
			foreach ($additionalNotificationEmails as $additionalNotificationEmail)
			{
				if (!empty($additionalNotificationEmail) && filter_var($additionalNotificationEmail, FILTER_VALIDATE_EMAIL))
				{
					$recipients[] = $additionalNotificationEmail;
				}
			}
		}
		$emailContent = '';
		if (isset($asset->params['booking_approval_notification_email'])
			&&
			$asset->params['booking_approval_notification_email'] != '')
		{
			$emailContent = $asset->params['booking_approval_notification_email'];
		}

		$search       = array('{res_code}', '{asset_name}');
		$replace      = array($reservation->code, $asset->name);
		$emailContent = str_replace($search, $replace, $emailContent);

		$mail->setSender(array($config->get('mailfrom'), $config->get('fromname')));
		$mail->addRecipient($recipients);
		$mail->setSubject(JText::_('SR_FIELD_BOOKING_INQUIRY_NOTIFICATION_OWNER_EMAIL_SUBJECT'));
		$mail->setBody($emailContent);
		$mail->isHtml(false);

		return $mail->Send();
	}

	public function sendGenericReservationStatusChangeEmail($reservationId = null)
	{
		JModelLegacy::addIncludePath(JPATH_ADMINISTRATOR . '/components/com_solidres/models', 'SolidresModel');
		JLoader::register('SRUtilities', SRPATH_LIBRARY . '/utilities/utilities.php');
		$modelReservation = JModelLegacy::getInstance('Reservation', 'SolidresModel', array('ignore_request' => true));
		$modelAsset       = JModelLegacy::getInstance('ReservationAsset', 'SolidresModel', array('ignore_request' => true));
		$reservation      = $modelReservation->getItem($reservationId);
		$asset            = $modelAsset->getItem($reservation->reservation_asset_id);
		$mail             = JFactory::getMailer();
		$config           = JFactory::getConfig();

		$statuses = array(
			0  => JText::_('SR_RESERVATION_STATE_PENDING_ARRIVAL'),
			1  => JText::_('SR_RESERVATION_STATE_CHECKED_IN'),
			2  => JText::_('SR_RESERVATION_STATE_CHECKED_OUT'),
			3  => JText::_('SR_RESERVATION_STATE_CLOSED'),
			4  => JText::_('SR_RESERVATION_STATE_CANCELED'),
			5  => JText::_('SR_RESERVATION_STATE_CONFIRMED'),
			-2 => JText::_('JTRASHED')
		);

		$recipients                   = array();
		$recipients[]                 = $reservation->customer_email;
		$recipients[]                 = $asset->email;
		$additionalNotificationEmails = isset($asset->params['additional_notification_emails']) ? explode(',', $asset->params['additional_notification_emails']) : array();
		if (!empty($additionalNotificationEmails))
		{
			foreach ($additionalNotificationEmails as $additionalNotificationEmail)
			{
				$recipients[] = $additionalNotificationEmail;
			}
		}
		$emailContent = JText::sprintf('SR_RESERVATION_STATUS_CHANGE_EMAIL_CONTENT',
			($reservation->customer_firstname . ' ' . $reservation->customer_lastname),
			$reservation->code,
			$statuses[$reservation->state],
			$asset->name
		);

		$mail->setSender(array($config->get('mailfrom'), $config->get('fromname')));
		$mail->addRecipient($recipients);
		$mail->setSubject(JText::_('SR_RESERVATION_STATUS_CHANGE_EMAIL_SUBJECT'));
		$mail->setBody($emailContent);
		$mail->isHtml(false);

		return $mail->Send();
	}

	public function hasCheckIn($roomTypeID, $checkin)
	{
		$dbo   = JFactory::getDbo();
		$query = $dbo->getQuery(true);
		$query->select('count(*)')->from('#__sr_reservations AS a')
			->innerJoin('#__sr_rooms AS b ON b.room_type_id = ' . (int) $roomTypeID)
			->innerJoin('#__sr_reservation_room_xref AS c ON c.room_id = b.id AND c.reservation_id = a.id')
			->where('a.checkin = ' . $dbo->quote($checkin) . ' AND a.state != -2');

		return $dbo->setQuery($query)->loadResult() > 0;
	}
}
